#!/system/bin/sh

HELP="Usage: $0 [options] file1 [file2 ...]
options:
	-s --size BYTES		rotate if larger than X bytes (default 10485760 - 10MB)
	-k --keep N		keep N files (default: 5)
	-r --pre-run SCRIPT	script to be run before rotation
	-p --post-run SCRIPT	script to be run after rotation
	-d --delete-if-full	delete old logs and truncate existing ones if partition is full
	-m --min-avail PERCENT	minimum free space in the partition to trigger --delete-if-full (default: 10)
	-c --clean-only		only remove the KEEP+1..n logs, without rotating
	-n --no-act		do not act (for debugging)

SCRIPT can be 'executable -opts args %f' - %f is replaced with the name of the file currently working on.
"

OPTIONS=$(getopt -n "$0" -o hs:k:r:p:dm:cn --long "help,size:,keep:,pre-run:,post-run:,delete-if-full,min-avail:,clean-only,no-act" -- "$@")

if [ $? -ne 0 ] ; then
	exit 1
fi

if [ $# -lt 1 ] ; then
	echo -e "$HELP"
	exit 0
fi

eval set -- "$OPTIONS"

SIZE=10485760
KEEP=5
CLEAN_ONLY=0
NO_ACT=""
PRE_SCRIPT=""
POST_SCRIPT=""
DELETE_IF_FULL=0
MIN_AVAIL_PERCENT=10

while true
do
	case "$1" in
		-h|-help|--help)
			echo -e "$HELP"
			exit 0
			;;
		-s|--size)
			SIZE="$2"
			shift 2
			;;
		-k|--keep)
			KEEP="$2"
			shift 2
			;;
		-n|--no-act)
			NO_ACT="echo NO ACT; I would run: "
			shift
			;;
		-r|--pre-run)
			PRE_SCRIPT="$2"
			shift 2
			;;
		-p|--post-run)
			POST_SCRIPT="$2"
			shift 2
			;;
		-d|--delete-if-full)
			DELETE_IF_FULL=1
			shift
			;;
		-m|--min-avail)
			MIN_AVAIL_PERCENT="$2"
			shift 2
			;;
		-c|--clean-only)
			CLEAN_ONLY=1
			shift
			;;
		--)
			shift
			break
			;;
	esac
done

if [ $# -lt 1 ] ; then
	echo "missing file(s) to rotate"
	echo ""
	echo -e "$HELP"
	exit 1
fi

expr "$SIZE" + 0 > /dev/null 2>&1
check=$?
if [ -z "$SIZE" -o $check -ne 0 ] ; then
	echo "invalid size (must be an integer): $SIZE"
	exit 2
fi

expr "$KEEP" + 0 > /dev/null 2>&1
check=$?
if [ -z "$KEEP" -o $check -ne 0 ] ; then
	echo "invalid keep number: $KEEP"
	exit 3
fi

if [ $KEEP -lt 1 ] ; then
	echo "keep must be equal to 1 or greater"
	exit 3
fi

expr "$MIN_AVAIL_PERCENT" + 0 > /dev/null 2>&1
check=$?
if [ -z "$MIN_AVAIL_PERCENT" -o $check -ne 0 ] ; then
	echo "invalid min-avail (must be 0-100): $MIN_AVAIL_PERCENT"
	exit 4
fi

if [ "$MIN_AVAIL_PERCENT" -lt 0 -o "$MIN_AVAIL_PERCENT" -gt 100 ] ; then
	echo "invalid min-avail (must be 0-100): $MIN_AVAIL_PERCENT"
	exit 4
fi

function partitionIsFull() {
	fname="$@"
	availableBlocks="$(($(stat -f -c "%a*%S" "$fname" 2> /dev/null)))"
	totalBlocks="$(($(stat -f -c "%b*%S" "$fname" 2> /dev/null)))"
	if [ $totalBlocks -eq 0 ] ; then
		echo "unable to read total blocks"
		return 1
	fi
	freePercent="$(expr \( $availableBlocks \* 100 \) / $totalBlocks)"
	if [ $freePercent -le $MIN_AVAIL_PERCENT ] ; then
		echo "free space (${freePercent}%) equal or less then threshold (${MIN_AVAIL_PERCENT}%)"
		return 0
	fi
	return 1
}

function mustRotate() {
	fname="$@"
	if [ ! -f "$fname" ] ; then
		echo "file \"$fname\" does not exist"
		return 100
	fi
	file_size="$(stat -c '%s' "$fname")"
	if [ "$file_size" -lt $SIZE ] ; then
		echo "no need to rotate \"$fname\""
		return 1
	fi
	echo "need to rotate \"$fname\""
	return 0
}

function removeAllLogs() {
	fname="$@"
	echo "removing all logs of $fname"
	runPre "$fname"
	$NO_ACT rm -f "$fname" "${fname}."*.gz
	ret=$?
	runPost "$fname"
	return $ret
}

function removeOldLogs() {
	fname="$@"
	echo "keeping $KEEP old logs of $fname"
	count=$KEEP
	while true
	do
		oldLog="${fname}.${count}.gz"
		if [ ! -f "$oldLog" ] ; then
			break
		fi
		$NO_ACT rm -f "$oldLog"
		count=$((count + 1))
	done
	return 0
}

function shiftOldLogs() {
	fname="$@"
	echo "shifting old logs of $fname"
	count=$(($KEEP - 1))
	while test $count -ge 1
	do
		oldLog="${fname}.${count}.gz"
		if [ ! -f "$oldLog" ] ; then
			count=$((count - 1))
			continue
		fi
		destFile="$(echo "$oldLog" | sed "s@\.$count\.gz@.$(($count + 1)).gz@")"
		$NO_ACT mv -f "$oldLog" "$destFile"
		count=$((count - 1))
	done
	return 0
}

function rotateFile() {
	fname="$@"
	echo "rotating $fname"
	runPre "$fname"
	$NO_ACT mv -f "$fname" "$fname.1"
	ret=$?
	runPost "$fname"
	return $ret
}

function runPre() {
	fname="$@"
	if [ -n "$PRE_SCRIPT" ] ; then
		echo "running pre script for $fname"
		preScript="$(echo "$PRE_SCRIPT" | sed "s@%f@\"$fname\"@g")"
		$NO_ACT eval $preScript
	fi
}

function runPost() {
	fname="$@"
	if [ -n "$POST_SCRIPT" ] ; then
		echo "running post script for $fname"
		postScript="$(echo "$POST_SCRIPT" | sed "s@%f@\"$fname\"@g")"
		$NO_ACT eval $postScript
	fi
}

function compressFile() {
	fname="$@"
	echo "compressing ${fname}.1"
	$NO_ACT gzip "$fname.1"
	ret=$?
	return $ret
}


exitCode=0
for fname in "$@"
do
	if [ $CLEAN_ONLY -eq 1 ] ; then
		removeOldLogs "$fname"
		continue
	fi
	if [ $DELETE_IF_FULL -eq 1 ] ; then
		partitionIsFull "$fname"
		fullPart=$?
		if [ $fullPart -eq 0 ] ; then
			removeAllLogs "$fname"
			eRet=$?
			if [ $eRet -ne 0 ] ; then
				exitCode=$eRet
			fi
			continue
		fi
	fi
	mustRotate "$fname"
	toRotate=$?
	if [ $toRotate -eq 0 ] ; then
		removeOldLogs "$fname"
		shiftOldLogs "$fname"
		rotateFile "$fname"
		rRet=$?
		if [ $rRet -eq 0 ] ; then
			compressFile "$fname"
			cRet=$?
			if [ $cRet -ne 0 ] ; then
				exitCode=10
			fi
		elif [ $rRet -ne 100 ] ; then
			exitCode=10
		fi
	elif [ $toRotate != 1 ] ; then
		exitCode=10
	fi
done

if [ $exitCode -eq 0 ] ; then
	echo "all operations completed succesfully"
else
	echo "rotation of some files failed"
fi

exit $exitCode

